(* (C) 1999-2004                                                 *)
(* Cuihtlauac Alvarado, France Telecon, Recherche & Developement *)
(* Jean-Franois Monin, Universit Joseph Fourier - VERIMAG      *)

(* $Id: otags.ml,v 1.31 2008-03-26 15:04:43 tews Exp $ *)

module Store = functor (T : sig val init : string list end) -> struct

  let buffer = ref (List.rev T.init)

  (* list_fifo gives FIFO behaviour: 
   * elements add'ed first are at the start of the list
   *)
  let list_fifo () = List.rev !buffer

  (* list_fifo gives LIFO behaviour: 
   * elements add'ed last are at the start of the list
   *)
  let list_lifo () = !buffer

  let clear () = buffer := []

  let add s = buffer := s :: !buffer

  let del s =
    let rec loop = function
      | [] -> []
      | h :: t -> if h = s then loop t else h :: loop t
    in buffer := loop !buffer

  let map f = buffer := List.map f !buffer

end

module Suffix = Store(struct let init = [ ".mli"; ".ml" ] end)
module Target = Store(struct let init = [] end)
module Camlp4 = Store(struct let init = [] end)
module Bindir = Store(struct let init = [ Conf.bin ] end)
module Libdir = Store(struct let init = [ Conf.lib ] end)

(* Parsing arguments *)

let release = Conf.name ^ " " ^ Conf.version

let default x = function None -> x | Some x -> x

let the = function 
  | None -> assert false
  | Some x -> x

let usage = 
  let list_to_string u = 
    let str = List.fold_left (fun u v -> u ^ v ^ "; ") "[" u in
    String.sub str 0 (max (String.length str - 2) 1) ^ "]" in
  let print_flag f x = if f x then "(default)" else "" in
  let suffix_list = list_to_string (Suffix.list_fifo ()) in
  let parser_list = list_to_string (Camlp4.list_fifo ()) in 
  let bindir_list = list_to_string (Bindir.list_lifo ()) in 
  let libdir_list = list_to_string (Libdir.list_lifo ()) in [
    (
      "-v",
      Arg.Set Conf.verbose,
      "            Verbose, display debug messages"
    ); (
      "-o",
      Arg.String 
	(fun str -> Conf.output := (if str = "-" then None else (Some str))),
      let str = match !Conf.output with None -> "stdout" | Some x -> x in
        "<str>       Output tags file (default: " ^ str 
	^ ") (use - for standard output)"
    ); (
      "-sc",
      Arg.Unit Suffix.clear,
      "           Suffix list: clear (default: " ^ suffix_list ^ ")"
    ); (
      "-sa",
      Arg.String Suffix.add,
      "<str>      Suffix list: add (default: " ^ suffix_list ^ ")"
    ); (
      "-sr",
      Arg.String Suffix.del,
      "<str>      Suffix list: del (default: " ^ suffix_list ^ ")"
    ); (
      "-r",
      Arg.Set Conf.recursive,
      let set = print_flag (fun x -> x) !Conf.recursive in 
      "            Enable recursive directory search " ^ set
    ); (
      "-dr",
      Arg.Clear Conf.recursive,
      let set = print_flag (not) !Conf.recursive in 
      "           Disable recursive directory search " ^ set
    ); (
      "-vi",
      Arg.Unit (fun _ -> Conf.editor := Conf.Vi),
      let str = if !Conf.editor = Conf.Emacs then "" else "(default)" in
      "           Generate vi/vim tags " ^ str
    ); (
      "-emacs",
      Arg.Unit (fun _ -> Conf.editor := Conf.Emacs),
      let str = if !Conf.editor = Conf.Emacs then "(default)" else "" in
      "        Generate emacs tags " ^ str
    ); (
      "-camlp4",
      Arg.String (fun s -> Conf.camlp4 := Some s),
      "<str>  Camlp4 command (default: " ^
	(match !Conf.camlp4 with
	   | None -> "chosen from bindir at runtime"
	   | Some s -> s
	) ^
	")"
    ); (
      "-pc",
      Arg.Unit Camlp4.clear,
      "           Camlp4 parser list: clear (default: " ^ parser_list ^ ")"
    ); (
      "-pa",
      Arg.String Camlp4.add,
      "<str>      Camlp4 parser list: add (default: " ^ parser_list ^ ")"
    ); (
      "-pr",
      Arg.String Camlp4.del,
      "<str>      Camlp4 parser list: del (default: " ^ parser_list ^ ")"
    ); (
      "-q",
      Arg.Set Conf.quotations,
      let set = print_flag (fun x -> x) !Conf.quotations in 
      "            Accept quotations in input " ^ set
    ); (
      "-dq",
      Arg.Clear Conf.quotations,
      let set = print_flag (not) !Conf.quotations in 
      "           Reject quotations in input " ^ set
    ); (
      "-n",
      Arg.Set Conf.native,
      let set = print_flag (fun x -> x) !Conf.native in 
      "            Enable native camlp4 " ^ set 
    ); (
      "-dn",
      Arg.Clear Conf.native,
      let set = print_flag (not) !Conf.native in 
      "           Disable native camlp4 " ^ set 
    ); (
      "-mli-tags",
      Arg.Set Conf.mli_tags,
      let set = print_flag (fun x -> x) !Conf.mli_tags in
      "           Enable tags for interfaces " ^ set 
    ); (
      "-no-mli-tags",
      Arg.Clear Conf.mli_tags,
      let set = print_flag (not) !Conf.mli_tags in
      "           Disable tags for interfaces " ^ set 
    ); (
      "-bindir",
      Arg.String Bindir.add,
      "<str>      add dir to search for camlp4 (default: " ^ bindir_list ^ ")"
    ); (
      "-libdir",
      Arg.String Libdir.add,
      "<str>      add dir to search for camlp4 modules (default: " 
      ^ libdir_list ^ ")"
    ); (
      "-version",
      Arg.Unit (fun _ -> print_endline release),
      "      Display otags version number (" ^ release ^ ")"
    );
  ]

let pr_tags = function 
  | Conf.Emacs -> "pr_emacs_tags" 
  | Conf.Vi -> "pr_vi_tags"

let ncamlp4_name = function
  | Conf.Emacs -> "camlp4o_pr_emacs"
  | Conf.Vi -> "camlp4o_pr_vi"

let locate access path filename = 
  let rec loop = function
    | [] -> filename
    | prefix :: pos_list ->
        let path = prefix ^ "/" ^ filename in
        try (Unix.access path [ Unix.F_OK; access ]; path)
        with Unix.Unix_error _ -> loop pos_list
  in loop path
        
let arg_post_process _ =
    (* 
     * Note that 
     *   Conf.camlp4 := "camlp4o.opt"
     * is nonsense, because natively compiled camlp4 parsers
     * cannot load any extentions (because native ocaml programs 
     * cannot load any object files).
     * 
     * Therefore we also have to check Camlp4.list to see 
     * if the user added modules before we enable the native camlp4.
     *)
  if !Conf.native && !Camlp4.buffer <> [] then 
    Conf.native := false;
  if !Conf.native && !Conf.camlp4 = None
  then 
    Conf.camlp4 := Some(locate Unix.X_OK (Bindir.list_lifo())
			  (ncamlp4_name !Conf.editor))
  else
    begin
      Camlp4.add "tags.cma";
      Camlp4.add ((pr_tags !Conf.editor) ^ ".cma");
      Camlp4.map (locate Unix.R_OK (Libdir.list_fifo()));
    end;
  (match !Conf.camlp4 with
     | Some _ -> ()
     | None -> Conf.camlp4 := Some (locate Unix.X_OK (Bindir.list_lifo())
				      Conf.camlp4_default)
  );
  if !Conf.quotations then Camlp4.add "-with-quotations";
  if not !Conf.mli_tags then Camlp4.add "-mli-only-module"
    

let debug prog args ofile =
  if !Conf.verbose then begin
    List.iter (Printf.printf "%s ") (prog :: args);
    begin match ofile with
      | None -> ()
      | Some filename -> Printf.printf "> %s " filename 
    end;
    print_newline ();
    flush stdout
  end

let system prog args ofile =
  let _ = debug prog args ofile in
  let fd = match ofile with
    | None | Some "-" -> Unix.stdout
    | Some filename -> Unix.descr_of_out_channel (open_out filename) in
  let args = Array.of_list (prog :: args) in 
  let _ = Unix.create_process prog args Unix.stdin fd Unix.stderr in
  let _ = Unix.wait () in
  ()

let process_file (src_file, tag_file) =
  (* 
   * if !Conf.verbose then
   *   Printf.printf "process file %s\n" src_file;
   *)
  system (the !Conf.camlp4) 
    (Camlp4.list_fifo () @ ["-o"; tag_file; src_file]) None
  
let ls (dir : string) =
  let module Dir = Store(struct let init = [] end) in
  let dir_hdl = Unix.opendir dir in
  begin
    try while true do
      let s = Unix.readdir dir_hdl in
      if s.[0] != '.' then Dir.add (dir ^ "/" ^ s)
    done
    with End_of_file -> Unix.closedir dir_hdl
  end;
  Dir.list_fifo ()
      
let filename_compare filename1 filename2 = 
  let rec first_suffix = function
    | [] -> 0
    | suffix :: suffix_list -> 
        if Filename.check_suffix filename1 suffix then -1 
        else if Filename.check_suffix filename2 suffix then 1
        else first_suffix suffix_list in
  let chop filename = 
    try Filename.chop_extension filename with _ -> filename in
  let c = Pervasives.compare (chop filename1) (chop filename2) in
  if c = 0 then 
    if filename1 = filename2 then 0
    else first_suffix (Suffix.list_fifo ())
  else c
    

let rec expand filter buf arg = 
  try
    match (Unix.LargeFile.stat arg).Unix.LargeFile.st_kind with
      | Unix.S_REG ->
	  if filter arg then arg :: buf else buf
      | Unix.S_DIR -> 
	  if !Conf.recursive then expand_list filter buf (ls arg) else buf
      | Unix.S_CHR 
      | Unix.S_BLK 
      | Unix.S_LNK 
      | Unix.S_FIFO
      | Unix.S_SOCK -> 
	  Printf.eprintf "Warning: Ignore special file %s\n" arg;
	  buf
  with
    | Unix.Unix_error(err, "stat", file) ->
	Printf.printf "Warning: Ignore %s (%s)\n"
	  file (Unix.error_message err);
	buf

and expand_list filter buf = function
  | [] -> buf
  | arg :: arg_list -> expand_list filter (expand filter buf arg) arg_list

let main _ = 
  Arg.parse (Arg.align usage) Target.add "Available commands:";
  arg_post_process ();
  let filter arg = 
    List.exists (Filename.check_suffix arg) (Suffix.list_fifo ()) in
  let arg_list = expand_list filter [] (Target.list_fifo ()) in
  let arg_list = List.sort filename_compare arg_list in
  let tmp_list = List.map (fun s -> Filename.temp_file "otags" "") arg_list in
  List.iter process_file (List.combine arg_list tmp_list);
  system "cat" ("/dev/null" :: tmp_list) !Conf.output;
  system "rm" ("-f" :: tmp_list) None

let error s = begin
  prerr_string s;
  flush stderr
end

let _ = try main () with 
  | Unix.Unix_error (code, funct, arg) -> begin
      error (Conf.name ^ ": Unix error\n");
      error ("  Syscall: " ^ funct ^ "\n");
      error ("  Arg:     " ^ arg ^ "\n");
      error ("  Error:   " ^ Unix.error_message code ^ "\n");
    end
  | _ as e -> begin
      error (Conf.name ^ ": Fatal error\n");
      raise e;
    end 

