(* Otags reloaded
 * 
 * Hendrik Tews Copyright (C) 2010
 * 
 * This file is part of "Otags reloaded".
 * 
 * "Otags reloaded" is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * "Otags reloaded" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with "Otags reloaded". If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * $Id: otags.ml,v 1.10 2010-08-24 13:49:09 tews Exp $
 * 
 * main module with main function
 * 
 *)

open Conf
open Global
open Types
open Camlp4_names
open Parser_factory

module Loc = Camlp4.PreCast.Loc
module U = Unix
module UL = Unix.LargeFile

exception Skip_entry


let parser_error_message loc msg =
  prerr_endline (Loc.to_string loc);
  prerr_string "Camlp4 parse error: ";
  prerr_endline msg


let open_input_file file =
  try
    open_in file 
  with
    | Sys_error msg ->
      if not !silent then
	prerr_endline msg;
      raise Skip_entry

let camlp4_printer_option = ["-printer"; "d"]

let make_camlp4_command unit file =
  let (camlp4, rest_parsers) = camlp4_variant !parser_list in
  let camlp4 = Filename.concat camlp4_path camlp4 in
  let dir_options =
    List.fold_right (fun d res -> "-I" :: d :: res) !camlp4_search_path [] in
  let parser_options = 
    List.fold_right (fun p res -> "-parser" :: p :: res) rest_parsers [] in
  let unit_option = match unit with
    | Signature -> "-intf"
    | Structure -> "-impl"
  in
  let options = 
    camlp4 :: camlp4_printer_option 
    @ dir_options 
    @ parser_options 
    @ [unit_option; file]
  in
  String.concat " " options


let read_camlp4_dump ic file unit =
  let magic = match unit with
    | Signature -> Camlp4_config.camlp4_ast_intf_magic_number
    | Structure -> Camlp4_config.camlp4_ast_impl_magic_number
  in
  let magic_len = String.length magic in
  let input_magic = String.create magic_len 
  in
  try
    really_input ic input_magic 0 magic_len;
    if input_magic = magic then 
      match unit with
	| Signature -> Sig_ast(input_value ic : sig_item_t)
	| Structure -> Struct_ast(input_value ic : str_item_t)
    else 
      raise Skip_entry
  with
    | Skip_entry 
    | End_of_file ->
      Printf.eprintf "Error while reading the camlp4 ast for %s\n" file;
      raise Skip_entry
    
    

let directive_handler x = Some x

let parse_file_internally unit file =
  let pa = update_syntax() in
  if !verbose then
    Printf.eprintf "Process %s internally as %s\n" 
      file (string_of_unit_type unit);
  let start_loc = pa.mkloc file in
  let ic = open_input_file file in
  let stream = Stream.of_channel ic in
  let comp_unit = match unit with
    | Signature -> 
      (try
	 Sig_ast(pa.parse_interf ~directive_handler start_loc stream)
       with
	 | Loc.Exc_located(loc, Stream.Error msg) ->
	   parser_error_message loc msg;
	   Tags.empty_sig_ast
      )
    | Structure -> 
      (try
	 let ast = pa.parse_implem ~directive_handler start_loc stream in
	 (* 
          * let module Dump = 
	  *       Camlp4.Printers.DumpCamlp4Ast.Make(Camlp4.PreCast.Syntax)
	  * in
	  * Dump.print_implem ~input_file:file ~output_file:"camlp4-dump" ast;
          *)
	 Struct_ast ast
       with
	 | Loc.Exc_located(loc, Stream.Error msg) ->
	   parser_error_message loc msg;
	   Tags.empty_str_ast
      )
  in
  (comp_unit, ic)


let parse_file_externally unit file = 
  let camlp4 = make_camlp4_command unit file in
  if !verbose then begin
    Printf.eprintf "Parse %s externally as %s\n"
      file (string_of_unit_type unit);
    prerr_endline camlp4;
  end;
  let ast_ic = U.open_process_in camlp4 in
  let comp_unit = read_camlp4_dump ast_ic file unit in
  match U.close_process_in ast_ic with
    | U.WEXITED 0 -> comp_unit
    | _ -> 
      if not !silent then
	Printf.eprintf "External camlp4 on %s failed\n" file;
      raise Skip_entry


let process_file tagfun unit file =
  let (comp_unit, ic) = match !use_internal_parsers with
    | true -> parse_file_internally unit file
    | false -> (parse_file_externally unit file, open_input_file file)
  in
  tagfun.start_unit file ic;
  Tags.generate_tags ic tagfun.write_tag comp_unit;
  tagfun.finish_unit();
  close_in ic


let is_directory f explicitly_listed =
  try
    (UL.stat f).UL.st_kind = U.S_DIR
  with
    | U.Unix_error(error, _, _) ->
      if explicitly_listed && (not !silent) || !verbose then
	Printf.eprintf "stat failure on %s: %s\n"
	  f
	  (U.error_message error);
      raise Skip_entry


(* process file or directory f, generating output for tags_out_ref.
 * If subdir_ref_option <> None than process_entry has been called
 *      recursively from process_directory. In this case it must append 
 *      detected directories to the reference in subdir_ref_option. These 
 *      directories will be processed by process_directory later.
 * If subdir_ref_option = None than process_entry has been called
 *      from somewhere higher up. In this case it must invoke 
 *      process_directory 
 *      when it detects an directory (and option -r was present).
 *)
let rec process_entry tagfun f subdir_ref_option =
  if Filename.check_suffix f ".ml"
  then process_file tagfun Structure f
  else if Filename.check_suffix f ".mli"
  then process_file tagfun Signature f
  else if !recurse_subdirectories && is_directory f (subdir_ref_option = None)
  then 
    match subdir_ref_option with
      | None -> process_directory tagfun f
      | Some r -> 
	(* process_directory tagfun tags_out f *)
	r := f :: !r
  else if ((subdir_ref_option = None) && (not !silent)) || !verbose
  then Printf.eprintf "Skip unrecognized file %s\n" f
  else ()

and process_directory tagfun subdir =
  if !verbose then
    Printf.eprintf "Descend into directory %s\n" subdir;
  let subdirs = ref [] in
  let subdir_ref_option = Some subdirs in
  let handle = U.opendir subdir in
  let not_finished = ref true in
  while !not_finished do
    match 
      try Some(U.readdir handle) with End_of_file -> None
    with
      | Some ("." | "..") -> ()
      | Some entry ->
	(try
	   process_entry tagfun
	     (Filename.concat subdir entry)
	     subdir_ref_option	
	 with
	   | Skip_entry -> ()
	)
      | None ->
	not_finished := false
  done;
  List.iter (process_directory tagfun) !subdirs



let switch_to_external_parser () =
  use_internal_parsers := false;
  switched_to_external_by_hand := true;
  if !verbose then
    prerr_endline "Switch to external mode"


let switch_to_internal_parser () =
  use_internal_parsers := true;
  if !verbose then
    prerr_endline "Switch to internal mode"


let print_parser_list () =
  if !verbose then flush stderr;
  Printf.printf "Current parser list: %s\n%!"
    (String.concat ", " !parser_list)


let reset_parser_list () =
  parser_list := default_parser_list;
  if !verbose then
    prerr_endline "Reset parser list to default";
  if !switched_to_external_by_hand = false && !use_internal_parsers = false
  then begin
    use_internal_parsers := true;
    if !verbose then
      prerr_endline "Switch back to internal mode";
  end


(* Add command line argument parser_name to the parser list. Use 
 * canonical camlp4 names only and obey dependencies. Do not add a 
 * given parser more than once.
 *)
let add_parser parser_name =
  let use_internal_before = !use_internal_parsers in
  let parsers_to_add = ref []
  in
    (try
       parsers_to_add := parser_name_and_dependency parser_name
     with
       | Not_found ->
	   use_internal_parsers := false;
	   parsers_to_add := [parser_name]
    );
    parsers_to_add := 
      List.filter (fun pa -> not (List.mem pa !parser_list)) !parsers_to_add;
    if !verbose then begin
      if !parsers_to_add = [] then
	prerr_endline "Add parser(s): all parsers already present in parser list"
      else
	Printf.eprintf "Add parser(s) %s\n" (String.concat " " !parsers_to_add);
      if use_internal_before <> !use_internal_parsers then
	print_endline "Switch to external mode"
    end;
    parser_list := !parser_list @ !parsers_to_add


let clear_parser_list () =
  parser_list := [];
  if !verbose then
    prerr_endline "Clear parser list";
  if !switched_to_external_by_hand = false then begin
    use_internal_parsers := true;
    if !verbose then
      prerr_endline "Switch back to internal mode";
  end


(* Print version and exit *)
let print_version () =
  Printf.printf "otags version %s for ocaml %s.x compiled with ocaml %s\n"
    otags_version ocaml_version Sys.ocaml_version;
  exit 0
      
  
type otags_actions =
  | Clear_parser_list
  | Add_parser of string
  | Reset_parser_list
  | Print_parser_list
  | Use_internal_parsers
  | Use_external_parsers
  | Process_file of string


let action_list = ref []

let queue_action a () = 
  action_list := a :: !action_list

let add_to_camlp4_search_path d =
  camlp4_search_path := !camlp4_search_path @ [d]


let anon_fun s = queue_action(Process_file s) ()

let arguments = Arg.align [
  ("-r", Arg.Set recurse_subdirectories,
   " descend recursively into directories");
  ("-o", Arg.String (fun f -> tags_file_name := Some f),
   "file output file [default TAGS for Emacs mode and tags for vi mode]");
  ("-a", Arg.Set append_to_tags_file,
   " append to an existing TAGS file");
  ("-vi", Arg.Clear emacs_mode,
   " generate tags for vi");
  ("-pc", Arg.Unit(queue_action Clear_parser_list),
   " clear parser list");
  ("-pa", Arg.String(fun s -> queue_action (Add_parser s) ()),
   "parser add parser to parser list");
  ("-pr", Arg.Unit(queue_action Reset_parser_list),
   " reset parser list to default [ocaml without extensions]");
  ("-pp", Arg.Unit(queue_action Print_parser_list),
   " print current parser list");
  ("-I", Arg.String add_to_camlp4_search_path,
   "dir add directory dir to the camlp4 search path for object files");
  ("-intern", Arg.Unit(queue_action Use_internal_parsers),
   " switch to internal mode [internal mode is default]");
  ("-extern", Arg.Unit(queue_action Use_external_parsers),
   " switch to external mode");
  ("-version", Arg.Unit print_version,
   " print version and exit");
  ("-v", Arg.Unit(fun () -> verbose := true; silent := false),
   " be more verbose");
  ("-q", Arg.Unit(fun () -> verbose := false; silent := true),
   " be quiet");
]


let usage_message =
  Printf.sprintf 
    "Usage %s [arguments...]\n\
     Creates tags files for Emacs or vi from Ocaml sources.\n\
     Options and arguments can be mixed. Order matters.\n\
     Options apply only to the arguments behind them.\n\
     Recognized options:"
    Sys.argv.(0)


let run_action tag_fun = function
  | Clear_parser_list -> clear_parser_list () 
  | Add_parser parser_name -> add_parser parser_name
  | Reset_parser_list -> reset_parser_list ()
  | Print_parser_list -> print_parser_list ()
  | Use_internal_parsers -> switch_to_internal_parser ()
  | Use_external_parsers -> switch_to_external_parser ()
  | Process_file file -> 
    try
      process_entry tag_fun file None
    with
      | Skip_entry -> ()


let main () =
  Arg.parse arguments anon_fun usage_message;
  if !append_to_tags_file && !emacs_mode = false then begin
    prerr_endline 
      "Appending to tags files is only supported for emacs TAGS files!";
    exit 1;
  end;
  let output_name = match !tags_file_name with
    | Some f -> f
    | None -> match !emacs_mode with
	| true -> "TAGS"
	| false -> "tags"
  in
  let tags_oc = 
    open_out_gen 
      (if !append_to_tags_file 
       then [Open_append; Open_creat; Open_text]
       else [Open_wronly; Open_trunc; Open_creat; Open_text])
      0o666 output_name
  in
  let tagfun = 
    if !emacs_mode 
    then Emacs.init tags_oc
    else Vi.init tags_oc
  in
  List.iter 
    (run_action tagfun)
    (List.rev !action_list);
  tagfun.finish_tagging();
  close_out tags_oc


let main_ex () =
  try
    if Array.length Sys.argv >= 2 && Sys.argv.(1) = "-v" then
      Printexc.record_backtrace true;
    main ()
  with
    | e -> 
      let backtrace = if !verbose then Printexc.get_backtrace() else "" in
      prerr_string "\nFatal error: escaping exception ";
      prerr_endline (Printexc.to_string e);
      (match e with
	| Loc.Exc_located(loc, oe) ->
	  prerr_endline (Loc.to_string loc);
	  prerr_endline (Printexc.to_string oe);
	| U.Unix_error(error, _func, _info) ->
	  Printf.eprintf "%s\n" (U.error_message error)      
	| _ -> ()
      );
      prerr_endline "";
      if !verbose then begin
	prerr_string backtrace;
	prerr_endline 
	  "\n\
           Please send the command line, the input files and the output\n\
           above as bug report to otags@askra.de";
      end
      else 
	prerr_endline 
	  "Please rerun otags with -v as *first* option to get a backtrace\n\
           and send the command line, the input files and the backtrace\n\
           as bug report to otags@askra.de";
      exit 2

;;

main_ex()
