(* Otags reloaded
 * 
 * Hendrik Tews Copyright (C) 2010
 * 
 * This file is part of "Otags reloaded".
 * 
 * "Otags reloaded" is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * "Otags reloaded" is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with "Otags reloaded". If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * $Id: parser_factory.ml,v 1.6 2010-08-24 11:21:31 tews Exp $
 * 
 * build new camlp4 parsers
 * 
 *)

open Global
open Types

type parser_functions = { 
  mkloc : string -> loc_t;
  parse_implem : 
    ?directive_handler:(str_item_t -> str_item_t option) -> 
		                       loc_t -> char Stream.t -> str_item_t;
  parse_interf : 
    ?directive_handler:(sig_item_t -> sig_item_t option) -> 
		                       loc_t -> char Stream.t -> sig_item_t;
}

let current_parser = ref {
  mkloc = (fun _ -> assert false);
  parse_implem = (fun ?directive_handler:_dh _ -> assert false);
  parse_interf = (fun ?directive_handler:_dh _ -> assert false);
}


(* functor for creating a fresh grammar *)
module FreshGrammar(Unit : sig end) 
  : Camlp4.Sig.Camlp4Syntax with module Loc = Camlp4.PreCast.Ast.Loc
                     and module Ast = Camlp4.PreCast.Ast
                     and module Token = Camlp4.PreCast.Gram.Token
                     and module Gram = Camlp4.PreCast.Gram
= Camlp4.OCamlInitSyntax.Make
    (Camlp4.PreCast.Ast)(Camlp4.PreCast.Gram)
    (Camlp4.Struct.Quotation.Make(Camlp4.PreCast.Ast))


(* List of camlp4 standard parsers with their long names.
 * All these are treated internally in the following function.
 * 
 * Camlp4OCamlRevisedParser
 * Camlp4OCamlReloadedParser
 * Camlp4OCamlParser
 * Camlp4OCamlRevisedParserParser
 * Camlp4OCamlParserParser
 * Camlp4GrammarParser
 * Camlp4MacroParser
 * Camlp4QuotationCommon
 * Camlp4QuotationExpander
 * Camlp4OCamlRevisedQuotationExpander
 * Camlp4OCamlOriginalQuotationExpander
 * Camlp4ListComprehension
 *)

let build_parser () =
  if !verbose then
    Printf.eprintf "Build new parser out of %s\n"
      (String.concat " " !parser_list);
  let module Otags_syntax = FreshGrammar(struct end) in
  let apply_parsing_extensions = function
    | "Camlp4OCamlRevisedParser" ->
	let module M = Camlp4OCamlRevisedParser.Make(Otags_syntax) in ()
    (* 
     * | "Camlp4OCamlReloadedParser" ->
     * 	let module M = Camlp4OCamlReloadedParser.Make(Otags_syntax) in ()
     *)
    | "Camlp4OCamlParser" ->
	let module M = Camlp4OCamlParser.Make(Otags_syntax) in ()
    | "Camlp4OCamlRevisedParserParser" ->
	let module M = Camlp4OCamlRevisedParserParser.Make(Otags_syntax) in ()
    | "Camlp4OCamlParserParser" ->
	let module M = Camlp4OCamlParserParser.Make(Otags_syntax) in ()
    | "Camlp4GrammarParser" ->
	let module M = Camlp4GrammarParser.Make(Otags_syntax) in ()
    | "Camlp4MacroParser" ->
	let module M = Camlp4MacroParser.Make(Otags_syntax) in ()
    | "Camlp4QuotationCommon" ->
      (* The Camlp4QuotationCommon module provides a Make functor, 
       * but this Make takes two arguments (so it does not fit the pattern 
       * here) and is not meant to be applied to any syntax when the module 
       * is loaded. Instead other quotation-building functors rely on 
       * this Make functor. Therefore we should not do anything here.
      *)
      ()
    | "Camlp4QuotationExpander" ->
	let module M = Camlp4QuotationExpander.Make(Otags_syntax) in ()
    | "Camlp4OCamlRevisedQuotationExpander" ->
      let module M =
	    Add_quotation.Make(Otags_syntax)(Camlp4OCamlRevisedParser.Make)
      in ()
    | "Camlp4OCamlOriginalQuotationExpander" ->
      let module OS = 
	    functor(S : Camlp4.Sig.Camlp4Syntax) ->
	      struct
		include 
		  Camlp4OCamlParser.Make(Camlp4OCamlRevisedParser.Make(S))
	      end
      in
      let module M =
	    Add_quotation.Make(Otags_syntax)(OS)
      in ()
    | "Camlp4ListComprehension" ->
	let module M = Camlp4ListComprehension.Make(Otags_syntax) in ()
    | _ -> assert false
  in
  List.iter apply_parsing_extensions !parser_list;
  let result = 
    { mkloc = Otags_syntax.Loc.mk;
      parse_implem = Otags_syntax.parse_implem;
      parse_interf = Otags_syntax.parse_interf;
    }
  in
    current_parser := result;
    result




let current_parser_list = ref []

let update_syntax () =
  if !current_parser_list == !parser_list 
  then 
    !current_parser
  else begin
    current_parser_list := !parser_list;
    build_parser ()
  end
